<?php
/**
 * admin_api.php - v16.7 核心修复版
 * 修复：自动创建缺失的 settings 表、支持商品发布、增强容错
 */
session_start();
require 'db.php';
header('Content-Type: application/json; charset=utf-8');
ini_set('display_errors', 0); 
error_reporting(0);

// 1. 安全拦截
if (!isset($_SESSION['admin_logged_in'])) { 
    echo json_encode(['status'=>'error', 'msg'=>'登录超时，请刷新页面重试']); exit; 
}

$action = $_POST['action'] ?? '';

// cURL 通用函数
function curl_req($url, $postData=[]) {
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $url);
    if(!empty($postData)) {
        curl_setopt($ch, CURLOPT_POST, 1);
        curl_setopt($ch, CURLOPT_POSTFIELDS, http_build_query($postData));
    }
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
    curl_setopt($ch, CURLOPT_TIMEOUT, 15);
    curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
    // 适配 HTTPS (如果服务器证书有问题可取消注释下一行)
    // curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
    $res = curl_exec($ch);
    $code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    return ['code'=>$code, 'body'=>$res];
}

try {
    // 🔥 [核心修复] 自动检测并创建 settings 表
    // 如果没有这张表，保存配置永远不会成功
    $conn->exec("CREATE TABLE IF NOT EXISTS `settings` (
        `config_key` varchar(50) PRIMARY KEY,
        `config_value` text
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;");

    // ==========================================
    // 业务逻辑区域
    // ==========================================

    // 1. 保存全局配置 (含 AI、站点名称)
    if ($action === 'save_settings') {
        foreach($_POST as $k=>$v) {
            if($k === 'action') continue;
            // 使用 REPLACE INTO 确保存在则更新，不存在则插入
            $stmt = $conn->prepare("REPLACE INTO settings (config_key, config_value) VALUES (?, ?)");
            $stmt->execute([$k, htmlspecialchars($v, ENT_QUOTES)]);
        }
        echo json_encode(['status'=>'success', 'msg'=>'配置已保存并生效']);
    }

    // 2. 后台发布商品 (新增功能)
    elseif ($action === 'admin_publish') {
        $title = trim($_POST['title']);
        $price = floatval($_POST['price']);
        $desc  = $_POST['desc'] ?? '';
        
        if(!$title || $price < 0) throw new Exception('商品名称和价格必填');

        // 默认绑定到管理员账号 (ID=1)
        // 如果想绑定到 AI 账号，可以读取 ai_user_id 配置
        $uid = 1; 
        
        $stmt = $conn->prepare("INSERT INTO products (user_id, title, price, description, status, created_at) VALUES (?, ?, ?, ?, 1, NOW())");
        $stmt->execute([$uid, $title, $price, $desc]);
        
        echo json_encode(['status'=>'success', 'msg'=>'商品发布成功']);
    }

    // 3. 获取授权信息 (透传 Source 站数据)
    elseif ($action === 'get_license_info') {
        $lock = @file_get_contents('license.lock');
        $key = 'Unknown';
        if(preg_match('/KEY:\s*(.+)/', $lock, $m)) $key = trim($m[1]);
        
        $res = curl_req("http://source.ictcode.com/api/query_license.php", [
            'key' => $key,
            'domain' => $_SERVER['HTTP_HOST']
        ]);
        
        if($res['code'] == 200) echo $res['body'];
        else echo json_encode(['status'=>'error', 'msg'=>'连接授权中心失败 (HTTP '.$res['code'].')']);
    }

    // 4. 更换授权码
    elseif ($action === 'change_license') {
        $newKey = trim($_POST['new_key']);
        if(!$newKey) throw new Exception('请输入新的授权码');

        // 远程验证
        $res = curl_req("http://source.ictcode.com/api/query_license.php", [
            'key' => $newKey,
            'domain' => $_SERVER['HTTP_HOST'],
            'ip' => $_SERVER['REMOTE_ADDR']
        ]);
        
        $data = json_decode($res['body'], true);
        if ($res['code'] == 200 && ($data['status']??'') === 'success') {
            // 验证通过，写入本地
            if(file_put_contents('license.lock', "KEY: $newKey")) {
                echo json_encode(['status'=>'success', 'msg'=>'激活成功！当前版本：'.$data['data']['type_text']]);
            } else {
                throw new Exception('写入文件失败，请检查目录权限');
            }
        } else {
            throw new Exception($data['msg'] ?? '激活失败，请检查授权码');
        }
    }

    // 5. 检查系统更新 (OTA)
    elseif ($action === 'check_update') {
        $current_ver = file_exists('version.php') ? include 'version.php' : '1.0.0';
        $res = curl_req("http://source.ictcode.com/ota/update.json");
        
        if ($res['code'] != 200) throw new Exception('无法连接更新服务器');
        
        $info = json_decode($res['body'], true);
        if ($info && isset($info['version']) && version_compare($info['version'], $current_ver, '>')) {
            echo json_encode([
                'status'=>'success', 
                'has_update'=>true, 
                'data'=> [
                    'version'=>$info['version'], 
                    'changelog'=>$info['changelog']
                ]
            ]);
        } else {
            echo json_encode(['status'=>'success', 'has_update'=>false, 'msg'=>'当前已是最新版本']);
        }
    }

    // 6. 执行在线更新
    elseif ($action === 'do_update') {
        // 1. 获取更新包地址
        $res = curl_req("http://source.ictcode.com/ota/update.json");
        $info = json_decode($res['body'], true);
        if (!$info || empty($info['download_url'])) throw new Exception('无法获取更新包地址');
        
        // 2. 下载文件
        $tempFile = __DIR__ . '/update_pkg.zip';
        $fp = fopen($tempFile, 'w+');
        $ch = curl_init($info['download_url']);
        curl_setopt($ch, CURLOPT_TIMEOUT, 300);
        curl_setopt($ch, CURLOPT_FILE, $fp);
        curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
        curl_exec($ch);
        if(curl_errno($ch)) { fclose($fp); throw new Exception('下载失败: '.curl_error($ch)); }
        curl_close($ch);
        fclose($fp);

        // 3. 解压覆盖
        $zip = new ZipArchive;
        if ($zip->open($tempFile) === TRUE) {
            $zip->extractTo(__DIR__);
            $zip->close();
            // 更新版本号
            file_put_contents('version.php', "<?php return '{$info['version']}'; ?>");
            @unlink($tempFile); // 删除临时文件
            echo json_encode(['status'=>'success', 'msg'=>'系统升级成功，即将刷新...']);
        } else {
            @unlink($tempFile);
            throw new Exception('解压失败，请检查网站目录是否有写入权限');
        }
    }

    // 7. 管理操作：删除帖子/商品、封禁用户
    elseif ($action === 'del_post') {
        $conn->prepare("DELETE FROM posts WHERE id=?")->execute([intval($_POST['id'])]);
        echo json_encode(['status'=>'success']);
    }
    elseif ($action === 'del_product') {
        $conn->prepare("DELETE FROM products WHERE id=?")->execute([intval($_POST['id'])]);
        echo json_encode(['status'=>'success']);
    }
    elseif ($action === 'toggle_ban') {
        $conn->prepare("UPDATE users SET status=? WHERE id=?")->execute([$_POST['status'], intval($_POST['id'])]);
        echo json_encode(['status'=>'success']);
    }

    // 8. 修改管理员密码
    elseif ($action === 'change_password') {
        $stmt = $conn->prepare("SELECT password FROM admins WHERE username='admin' LIMIT 1");
        $stmt->execute();
        $user = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if ($user && password_verify($_POST['old_pass'], $user['password'])) {
            $newHash = password_hash($_POST['new_pass'], PASSWORD_DEFAULT);
            $conn->prepare("UPDATE admins SET password=? WHERE username='admin'")->execute([$newHash]);
            echo json_encode(['status'=>'success', 'msg'=>'密码修改成功，请重新登录']);
        } else {
            echo json_encode(['status'=>'error', 'msg'=>'旧密码错误']);
        }
    }

    else { 
        echo json_encode(['status'=>'error', 'msg'=>'未知操作: ' . htmlspecialchars($action)]); 
    }

} catch (Exception $e) {
    // 捕获所有异常并返回 JSON
    echo json_encode(['status'=>'error', 'msg'=>'系统错误: '.$e->getMessage()]);
}
?>